%% Script that analyzes the reaction time data using a linear mixed effects model
clear,clc,close all

%% Statistical analysis
% Load the RT files
load('RT/RTNH_1');      RTNH_1 = dRT;
load('RT/RTNH_2');      RTNH_2 = dRT;
load('RT/RTNH_3');      RTNH_3 = dRT;
load('RT/RTUHL_1');     RTUHL_1 = dRT;
load('RT/RTUHL_2');     RTUHL_2 = dRT;
load('RT/RTUHL_3');     RTUHL_3 = dRT;       clear dRT

% Organize data in columns
RT = [reshape(RTNH_1,128,1);reshape(RTNH_2,128,1);reshape(RTNH_3,128,1);...
      reshape(RTUHL_1,128,1);reshape(RTUHL_2,128,1);reshape(RTUHL_3,128,1)];
Subj = nan(16*8,1);
for k=1:16, Subj((k-1)*8+1:k*8) = k*ones(8,1); end
Subj = repmat(Subj,3,1);
Subj = [Subj;Subj+16];
Cond = nominal(repmat([repmat('Quiet',128,1);repmat('Noise',256,1)],2,1));
Hearing = nominal([repmat('NHP',384,1) ; repmat('UHL',384,1)]);

% Create a table with the column vectors
tbl = table(RT,Cond,Hearing,Subj,'VariableNames',{'RT','Cond','Hearing','Subj'});

% Linear mixed-effects model
GLME = fitglme(tbl,'RT ~ Cond + Hearing + (1|Subj)','Distribution','Gamma','Link',-1);
disp(GLME)

% Print estimated values
fprintf('RT in NOISE for NH = %1.2f ms\n',...
    double(GLME.Coefficients(1,2))^(-1))
fprintf('RT in QUIET = %1.2f ms\n',...
    (double(GLME.Coefficients(1,2))+double(GLME.Coefficients(2,2)))^(-1))
fprintf('RT for UHL = %1.2f ms\n\n\n',...
    (double(GLME.Coefficients(1,2))+double(GLME.Coefficients(3,2)))^(-1))

%% Figure
figure('PaperSize',[20 16]),hold on,grid on,box on
cmap = colormap('lines');
sep = 0.1;      disp = 0.02;        MS = 7;       FS = 16;      LW = 1;

% Plot median RTs per participant
plot(1-sep+disp*randn(16,1),nanmedian(RTNH_1),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(1,:),'MarkerEdgeColor','w')
plot(1+sep+disp*randn(16,1),nanmedian(RTUHL_1),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(2,:),'MarkerEdgeColor','w')
plot(2-sep+disp*randn(16,1),nanmedian(RTNH_2),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(1,:),'MarkerEdgeColor','w')
plot(2+sep+disp*randn(16,1),nanmedian(RTUHL_2),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(2,:),'MarkerEdgeColor','w')
plot(3-sep+disp*randn(16,1),nanmedian(RTNH_3),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(1,:),'MarkerEdgeColor','w')
plot(3+sep+disp*randn(16,1),nanmedian(RTUHL_3),'o','MarkerSize',MS,...
    'MarkerFaceColor',cmap(2,:),'MarkerEdgeColor','w')

% Labels
set(gca,'xtick',1:3,'xticklabel',{'Quiet','Noise','Noise'},'FontSize',FS)
set(gca,'ytick',400:200:1600,'FontSize',FS)
axis([0.5 3.5 250 2200])
ylabel('Reaction time (ms)                    ','FontSize',FS)

% Median scores
text(1-0.38,350,sprintf('%1.0f ms',nanmean(nanmedian(RTNH_1))),'FontSize',FS-1)
text(1+0.05,370,sprintf('%1.0f ms',nanmean(nanmedian(RTUHL_1))),'FontSize',FS-1)
text(2-0.38,360,sprintf('%1.0f ms',nanmean(nanmedian(RTNH_2))),'FontSize',FS-1)
text(2+0.05,380,sprintf('%1.0f ms',nanmean(nanmedian(RTUHL_2))),'FontSize',FS-1)
text(3-0.38,420,sprintf('%1.0f ms',nanmean(nanmedian(RTNH_3))),'FontSize',FS-1)
text(3+0.05,440,sprintf('%1.0f ms',nanmean(nanmedian(RTUHL_3))),'FontSize',FS-1)

% Quiet vs Noise
sep = 0;
plot([0.7,1.3],[2000,2000]+sep,'LineWidth',LW,'Color','k')
plot([0.7,0.7],[1960,2000]+sep,'LineWidth',LW,'Color','k')
plot([1.3,1.3],[1960,2000]+sep,'LineWidth',LW,'Color','k')
text(1,1950+sep,'Quiet','FontSize',FS-1,...
    'HorizontalAlignment','center','color','k')
plot([1.7,3.3],[2000,2000]+sep,'LineWidth',LW,'Color','k')
plot([1.7,1.7],[1960,2000]+sep,'LineWidth',LW,'Color','k')
plot([3.3,3.3],[1960,2000]+sep,'LineWidth',LW,'Color','k')
text(2.5,1950+sep,'Noise','FontSize',FS-1,...
    'HorizontalAlignment','center','color','k')
plot([1.0,2.5],[2040,2040]+sep,'LineWidth',LW,'Color','k')
plot([1.0,1.0],[2000,2040]+sep,'LineWidth',LW,'Color','k')
plot([2.5,2.5],[2000,2040]+sep,'LineWidth',LW,'Color','k')
text(1.75,2080,'***','FontSize',FS+10,...
    'HorizontalAlignment','center','color','k')
text(3.45,2100,'RT_N \uparrow 111ms','FontSize',FS-2,...
    'HorizontalAlignment','right','color','k')

% NH vs UHL
plot([0.7,1.9],[1700,1700],'LineWidth',LW,'Color',cmap(1,:))
plot([0.7,0.7],[1660,1700],'LineWidth',LW,'Color',cmap(1,:))
plot([1.9,1.9],[1660,1700],'LineWidth',LW,'Color',cmap(1,:))
text(1.3,1650,'Normal hearing','FontSize',FS-1,...
    'HorizontalAlignment','center','color',cmap(1,:))
plot([2.1,3.3],[1700,1700],'LineWidth',LW,'Color',cmap(2,:))
plot([2.1,2.1],[1660,1700],'LineWidth',LW,'Color',cmap(2,:))
plot([3.3,3.3],[1660,1700],'LineWidth',LW,'Color',cmap(2,:))
text(2.7,1650,'Unilateral hearing loss','FontSize',FS-1,...
    'HorizontalAlignment','center','color',cmap(2,:))
plot([1.3,2.7],[1740,1740],'LineWidth',LW,'Color','k')
plot([1.3,1.3],[1700,1740],'LineWidth',LW,'Color','k')
plot([2.7,2.7],[1700,1740],'LineWidth',LW,'Color','k')
text(2,1775,'***','FontSize',FS+10,...
    'HorizontalAlignment','center','color','k')
text(3.45,1800,'RT_{UHL} \uparrow 103ms','FontSize',FS-2,...
    'HorizontalAlignment','right','color','k')

% print figure
orient tall
print('-dpng','-r600','Figure3')
print('-dpdf','-r600','Figure3')
close all



















